<?php
/* --------------------------------------------------------------
  OnGetSellingUnitEanEventListener.php 2020-02-21
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2020 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\Properties\SellingUnitEan\Listener;

use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\ProductModifiers\Modifiers\Collections\ModifierIdentifierCollectionInterface;
use Gambio\Shop\Properties\SellingUnitEan\Repository\Dto\CombiIdDto;
use Gambio\Shop\Properties\SellingUnitEan\Repository\Dto\EanDto;
use Gambio\Shop\Properties\SellingUnitEan\Service\ReadServiceInterface;
use Gambio\Shop\Properties\SellingUnitImages\Database\Exceptions\ModifierCollectionDoesNotContainAnyPropertiesException;
use Gambio\Shop\Properties\SellingUnitImages\Database\Repository\Helpers\CombisIdIdentifierInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSellingUnitEanEventInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\OnGetSellingUnitEanEvent;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Ean;

/**
 * Class OnGetSellingUnitEanEventListener
 * @package Gambio\Shop\Properties\SellingUnitEan\Listener
 */
class OnGetSellingUnitEanEventListener
{
    /**
     * @var ReadServiceInterface
     */
    protected $service;
    
    /**
     * @var CombisIdIdentifierInterface
     */
    protected $combisIdIdentifier;
    
    
    /**
     * OnGetSellingUnitEanEventListener constructor.
     *
     * @param ReadServiceInterface        $service
     * @param CombisIdIdentifierInterface $combisIdIdentifier
     */
    public function __construct(
        ReadServiceInterface $service,
        CombisIdIdentifierInterface $combisIdIdentifier
    ) {
        $this->service            = $service;
        $this->combisIdIdentifier = $combisIdIdentifier;
    }
    
    
    /**
     *
     * @param OnGetSellingUnitEanEventInterface $event
     *
     */
    public function __invoke(OnGetSellingUnitEanEventInterface $event)
    {
        $productId = $event->productId();
        $modifiers = $event->modifierIdentifiers();
        if (count($modifiers)) {
            $ean         = $event->ean()->value();
            $valueEanDto = $this->getValueEan($productId, $modifiers);
            if ($valueEanDto->value() !== '') {
                $ean = $valueEanDto->value();
            }
            $event->setEan(new Ean($ean));
        }
    }
    
    
    /**
     * @param ProductId                             $productId
     * @param ModifierIdentifierCollectionInterface $identifierCollection
     *
     * @return EanDto
     * @throws ModifierCollectionDoesNotContainAnyPropertiesException
     */
    protected function getValueEan(
        ProductId $productId,
        ModifierIdentifierCollectionInterface $identifierCollection
    ) : EanDto {
        try {
            $combiIdDto = $this->combisIdIdentifier->getCombisId($productId, $identifierCollection);
            $dto        = new CombiIdDto($combiIdDto->combisId()); // only to match service signature
        } catch (ModifierCollectionDoesNotContainAnyPropertiesException $e) {
            return new EanDto('');
        }
        
        return $this->service->getEanValueByPropertyCombination($dto);
    }
    
}